function [results, veh, prof, info] = vehicleSim(veh, mission_name, opt)
%vehicleSim Vehicle simulation
%
% vehicleSim(configuration_name, mission_name) loads the mission and 
%   vehicle data for the specified configuration, runs the simulation and 
%   evaluates postprocessing results, using a default set of options.
%
%   veh is a struct containing the vehicle data, previously generated with
%       createData.
%
%   mission_name is a string specifying the name of a sheet contained in the
%       data/mission/cycles.xlsx spreadsheet, which defines a vehicle speed vs
%       time mission profile.
%
%   vehicleSim(_, 'option1', value1, ..., 'optionN', value)
%     specifies additional options and information with
%     name/value pairs.
%
%     Valid properties are:
%       SOC_initial: the initial SOC for BEVs.
%       loadFraction: the payload as a fraction of the maximum allowable
%           payload (defined in the vehicle_body properties).
%       timestep: the simulation timestep
%       driverModel (logical): whether to enable the driver model.

arguments
    % Required positional arguments
    veh 
    mission_name {mustBeText}
    % Name-Value pair arguments
    opt.SOC_initial {mustBeNumeric, mustBeInRange(opt.SOC_initial, 0, 1)} = 0.85
    opt.loadFraction {mustBeNumeric, mustBeInRange(opt.loadFraction, 0, 1)} = []
    opt.payload {mustBeNumeric} = [];
    opt.timestep {mustBeNumeric} = 1
    opt.driverModel {mustBeA(opt.driverModel, "logical")} = true
    opt.regenBrakingFract = 0.6;
    opt.eRoad = [];
end

%% Prepare mission data
% Load the mission file
missionFile = fullfile('library', 'mission', mission_name);
missionData = load(missionFile);
veh.dt = opt.timestep; % discretization timestep, seconds

switch missionData.basis
    case "time"
        % Resample
        mission = resampleMission(missionData, veh.dt);

    case "distance"
        mission = missionData;

    otherwise
        error("The mission.basis property must be either ""time"" or ""distance"".")

end

if ~isempty(opt.eRoad)
    mission = createERoadFromVecto(mission, opt.eRoad);
end

%% Prepare vehicle data
% Evaluate vehicle parameters that are mission-dependent or payload-dependent

% Add payload to the curb mass
if ~isempty(opt.loadFraction) && ~isempty(opt.payload)
    error("You cannot specify both ""loadFraction"" and ""payload"".")
elseif ~isempty(opt.loadFraction) && isempty(opt.payload)
    payload = opt.loadFraction * veh.body.maxLoad;
elseif isempty(opt.loadFraction) && ~isempty(opt.payload)
    payload = opt.payload;
else
    payload = 0;
end
veh.body.mass = veh.body.curbMass + payload;
veh.loadFraction = opt.loadFraction;

% Evaluate weight-dependent RRC
if strcmp(veh.body.axleMethod, 'explicit')
    veh.body.rollResCoeff0 = veh.body.rollResCoeff0(veh.body.mass);
end

% Add regen braking fraction
if strcmp(veh.vehType, 'bev')
    veh.em.regenBrakingFract = opt.regenBrakingFract;
end

% Engine correction factors (VECTO-like)
if isfield(veh, "eng") && isfield(veh.eng, "corrFactor")
    if isfield(missionData, "weightingFactor")
        corrFactorCycle = veh.eng.corrFactor.urban * missionData.weightingFactor.urban + ...
            veh.eng.corrFactor.rural * missionData.weightingFactor.rural + ...
            veh.eng.corrFactor.motorway * missionData.weightingFactor.motorway;
    else
        corrFactorCycle = 1;
        warning("The engine specifies fuel correction factors, but the mission does not specify weighting factors. No fuel correction was applied to the mission.")
    end
    veh.eng.corrFactor.total = corrFactorCycle * veh.eng.corrFactor.coldHot * veh.eng.corrFactor.aftertreatment * veh.eng.corrFactor.LHV;    
end

% Set SOC bounds
if isfield(veh, 'batt')
    veh.batt.SOC_lb = 0;
    veh.batt.SOC_ub = 1;
end

% Design behavioral acceleration limits
switch veh.body.class
    case "LightDuty"
        veh.ctrl.accLimitUp = accelerationCurve([0 25 60 150]/3.6, [3, 3, 1, 1]);
        veh.ctrl.accLimitLo = accelerationCurve([0 50 60 150]/3.6, -[1.5, 1.5, 0.8, 0.8]);
    case {"RigidTruck", "Tractor"}
        veh.ctrl.accLimitUp = accelerationCurve([0 25 60 120]/3.6, [1, 1, 0.5, 0.5]);
        veh.ctrl.accLimitLo = accelerationCurve([0 50 60 120]/3.6, -[1, 1, 0.5, 0.5]);
end

% Design transmission logic
switch veh.vehType
    case 'conv'
        if ~isfield(veh.gb, 'ctrl')
            veh.gb.ctrl = designEffShiftLogic(veh);
        end
    case 'bev'
        if ~isfield(veh.gb, 'ctrl')
            veh.gb.ctrl = designBevShiftLogic(veh);
        end
end

% Evaluate vehicle speed limit
veh.maxSpd = vehMaxSpd(veh);

%% Initialize states
GN0 = 0;
brakeCmd0 = 0;
switch veh.vehType
    case 'conv'
        x = {GN0};
        u = {GN0, brakeCmd0};

        % Controllers state
        transmState = {0 0};
        
        % Vehicle model fun
        vehModel = @(x, u, mis, veh) convModel(x, u, mis, veh);

    case 'bev'
        % Initial SOC
        battSOC = opt.SOC_initial;
        x = {battSOC};
        if isfield(veh, 'cap')
            % Set the initial supercap SOC so that the supercap and battery
            % terminal are equal
            batt_ocv_initial = veh.batt.ocVolt(battSOC);
            cap_ocv = batt_ocv_initial;
            capSOC = cap_ocv ./ veh.cap.nomVolt;
            x{2} = {capSOC};
        end

        % Control
        u = {GN0};

        % Controllers state
        transmState = {0 0};

        % Vehicle model fun
        vehModel = @(x, u, mis, veh) bevModel(x, u, mis, veh);

end

% Initialize driver 
driver.state = "drive";

% driver.coasting.targetSpdDecisionFactor = griddedInterpolant([0 48 52 100]./3.6, [0 0 1 1]);
% driver.coasting.spdDropDecisionFactor = griddedInterpolant([-100 9 11 100]./3.6, [1 1 0 0]);
% driver.coasting.vehSpdMin = 50 / 3.6;

%% Run simulation
% Initialize time, distance, speed, counters
time = 0;
vehDist = 0;
n = 0;

switch missionData.basis
    case "time"
        driverModel = @driverModelBasic;

        vehSpd = mission.speed_kmh(1) / 3.6;
        
    case "distance"
        driverModel = @driverModelPredictive;

        vehSpd = mission.speed_kmh(vehDist) / 3.6;
        finalDist = mission.speed_kmh.GridVectors{1}(end);
        veh.dx = 1; % m
        stop_counter = 1;

        mission.finalDist = finalDist;
        
end

% Progress bar
fprintf('Progress:    ')

terminate = false;
stop_state = false;
% run the simulation
while ~terminate
    n = n+1;

    dyn(n).time = time;
    dyn(n).vehDist = vehDist;
    dyn(n).vehSpd = vehSpd;

    switch missionData.basis
        case "time"
            % Progress bar
            fprintf('%s%2d %%', ones(1,4)*8, floor(time/mission.time_s(end)*100));

            % Retrieve mission speed and gradient
            dyn(n).vehGrad = mission.grad_percent(n);
            if n < length(mission.speed_kmh)
                desSpd_next = mission.speed_kmh(n+1) ./ 3.6;
            else
                desSpd_next = 0;
            end

            % Desired acceleration
            dyn(n).vehAcc = (desSpd_next - vehSpd) ./  ( veh.dt );

        case "distance"
            % Progress bar
            fprintf('%s%2d %%', ones(1,4)*8, floor(vehDist/finalDist*100));

            % Retrieve mission speed and gradient
            dyn(n).vehGrad = mission.grad_percent(vehDist);
            desSpd_next = mission.speed_kmh(vehDist + veh.dx) ./ 3.6;
            if isfield(mission, "eRoad")
                dyn(n).eRoad = mission.eRoad(vehDist);
            end

            % Handle stops
            if stop_state
                if stop_time >= stopDuration
                    % Exit stop state
                    stop_state = false;
                else
                    desSpd_next = 0;
                    stop_time = stop_time + veh.dt;
                end
            else
                if ( stop_counter < length(mission.stops.distance) ) && vehDist >= ( mission.stops.distance(stop_counter) )
                    % Enter stop state
                    stop_state = true;
                    stopDuration = mission.stops.duration(stop_counter);
                    stop_counter = stop_counter + 1;
                    stop_time = 0;
                    desSpd_next = 0;
                end
            end

            % Desired acceleration
            % dyn(n).vehAcc = (desSpd_next.^2 - vehSpd.^2) ./  ( 2*veh.dx );
            dyn(n).vehAcc = (desSpd_next - vehSpd) ./  ( veh.dx );

    end

    % Driver model
    if opt.driverModel
        vehModelCur = @(u, mis) vehModel(x, u, mis, veh);
        [dyn(n), u] = driverModel(dyn(n), mission, vehModelCur, veh, u, driver);
    else
        u{2} = 0;
    end

    % Advance the simulation
    [x, ~, prof(n)] = vehModel(x, u, dyn(n), veh);

    % Transmission logic
    switch veh.vehType
        case 'conv'
            % Conventional vehicle control logic
            [GN, transmState] = effShiftTransmLogic(transmState, prof(n), dyn(n), veh);
            u{1} = GN;

        case 'bev'
            % BEV control logic
            [GN, transmState] = bevTransmLogic(transmState, prof(n), dyn(n), veh);
            % GN = minCurrTransmLogic(dyn(n), veh, x{1});
            u{1} = GN;
    end
  
    % Update time, distance and speed
    vehSpd = vehSpd + dyn(n).vehAcc * veh.dt;
    vehDist = vehDist + dyn(n).vehSpd * veh.dt + 0.5 * dyn(n).vehAcc .* (veh.dt^2);
    time = time + veh.dt;

    switch missionData.basis
        case "time"
            % Check elapsed time
            terminate = ( n >= length(mission.time_s) );

        case "distance"
            % Check whether the simulation is stuck
            if ~stop_state && ( dyn(n).vehSpd * veh.dt < 1 )
                stuckCounter = stuckCounter + veh.dt;
                if stuckCounter > 20
                    % Attempt to force the simulation by moving the vehcile
                    % by one dx forward
                    vehDist = vehDist + veh.dx;
                end
            else
                stuckCounter = 0;
            end

            % Check travelled distance
            terminate = vehDist > ( finalDist - veh.dx );
    end

end

% Progress bar
fprintf('%s%2d %%', ones(1,4)*8, 100);
fprintf('\n')

% Add vehDist and refSpd to the time profiles
vehDist = [dyn.vehDist];
switch missionData.basis
    case "time"
        refSpd = mission.speed_kmh / 3.6;
    case "distance"
        refSpd = mission.speed_kmh(vehDist) / 3.6;
end
prof = structAsgn(prof, refSpd, vehDist);

%% Process time profiles
results.dist = trapz([dyn(:).time], [dyn(:).vehSpd]);
switch veh.vehType
    case 'conv'
        results.fuel_consumption_kg = trapz([dyn(:).time], [prof(:).fuelFlwRate]); % kg
        results.fuel_consumption_l = results.fuel_consumption_kg ./ veh.eng.fuelDensity; % l
        results.fuel_economy_l_km = results.fuel_consumption_l ./ (results.dist.*1e-3); % l/km
        results.fuel_economy_kg_km = results.fuel_consumption_kg ./ (results.dist.*1e-3); % kg/km
        results.CO2ttw_g_km = (results.fuel_consumption_kg .* veh.eng.CO2ttwFactor) / (results.dist.*1e-3); % g/km
        EC = (results.fuel_consumption_kg*1e-3 * veh.eng.fuelLHV); % J
    case {'bev', 'bev_dyn'}
        EC = trapz([dyn.time], [prof.battOCVolt] .* [prof.battCurr]); % J
end

EC = EC/3600; % Wh
results.ECttw_Wh_km = EC / (results.dist.*1e-3); % Wh/km

warning(['Unfeas: # ' num2str(sum([prof.unfeas]))])

% Mission statistics
info.missionName = mission_name;
info.actualDistance = results.dist;
info.missionDistance = trapz([dyn(:).time], [dyn(:).vehSpd]);
% Average difference between mission and actual vehicle speed, m/s
info.avg_speed_deviation = trapz([dyn(:).time], [dyn(:).vehSpd] - [dyn(:).vehSpd]) / dyn(end).time;
% Maximum difference between mission and actual vehicle speed, m/s
info.max_speed_deviation = max([dyn(:).vehSpd] - [dyn(:).vehSpd]);
% Share of time spent with a deviation larger than 5 km/h
speed_unmet = abs([dyn(:).vehSpd] - [dyn(:).vehSpd]) > (5/3.6);
info.speed_deviation_timeshare = trapz([dyn(:).time], speed_unmet) / dyn(end).time;

end